// js/trainer_test.js
document.addEventListener('DOMContentLoaded', () => {
    const materialId = document.getElementById('materialId')?.value;
    if (!materialId) return;

    const testContainer = document.getElementById('testContainer');
    const loading = document.getElementById('loading');
    const questionsList = document.getElementById('questionsList');
    const nextBtn = document.getElementById('nextBtn');
    const prevBtn = document.getElementById('prevBtn');
    const submitBtn = document.getElementById('submitBtn');
    const progressText = document.getElementById('progressText');
    const resultContainer = document.getElementById('resultContainer');

    let questions = [];
    let curIndex = 0;
    let userAnswers = {}; // Map questionId -> answer

    // Fetch Questions
    fetch('test_endpoint.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ action: 'start_test', material_id: materialId })
    })
        .then(res => res.json())
        .then(data => {
            loading.classList.add('d-none');
            if (data.success && data.questions.length > 0) {
                questions = data.questions;
                renderQuestion(0);
                testContainer.classList.remove('d-none');
            } else {
                alert("Error generating test: " + (data.error || "No questions found."));
                window.location.href = 'dashboard.php';
            }
        })
        .catch(err => {
            console.error(err);
            alert("Network error.");
        });

    function renderQuestion(index) {
        if (index < 0 || index >= questions.length) return;
        curIndex = index;

        const q = questions[index];
        const savedAns = userAnswers[q.id] || null;

        let optionsHtml = '';
        q.options.forEach((opt, i) => {
            const checked = savedAns === opt ? 'checked' : '';
            const uniqueId = `opt_${q.id}_${i}`;

            optionsHtml += `
                <div class="form-check p-0 mb-3" style="position: relative;">
                    <input class="form-check-input visually-hidden" type="radio" name="q_${q.id}" value="${opt.replace(/"/g, '&quot;')}" id="${uniqueId}" ${checked}>
                    <label class="form-check-label d-block p-3 rounded-3 text-white transition-all" for="${uniqueId}" 
                           style="background: rgba(255,255,255,0.05); border: 1px solid var(--border-color); cursor: pointer; transition: all 0.2s;">
                        <div class="d-flex align-items-center">
                            <span class="custom-radio me-3" style="min-width: 20px; height: 20px; border-radius: 50%; border: 2px solid var(--text-secondary); display: inline-block; position: relative;"></span>
                            ${opt}
                        </div>
                    </label>
                </div>
            `;
        });

        questionsList.innerHTML = `
            <div class="mb-4 animation-fade">
                <h5 class="fw-bold text-white mb-4 line-height-base">${index + 1}. ${q.text}</h5>
                <div class="options-list">
                    ${optionsHtml}
                </div>
            </div>
        `;

        // Bind Change Events
        document.querySelectorAll(`input[name="q_${q.id}"]`).forEach(inp => {
            inp.addEventListener('change', (e) => {
                userAnswers[q.id] = e.target.value;
            });
        });

        updateControls();
    }

    function updateControls() {
        progressText.textContent = `Question ${curIndex + 1} of ${questions.length}`;
        prevBtn.disabled = curIndex === 0;

        if (curIndex === questions.length - 1) {
            nextBtn.classList.add('d-none');
            submitBtn.classList.remove('d-none');
        } else {
            nextBtn.classList.remove('d-none');
            submitBtn.classList.add('d-none');
        }
    }

    nextBtn.addEventListener('click', () => {
        // Validate answer selected? Maybe optional, but better required.
        const q = questions[curIndex];
        if (!userAnswers[q.id]) {
            alert("Please select an answer to proceed.");
            return;
        }
        renderQuestion(curIndex + 1);
    });

    prevBtn.addEventListener('click', () => renderQuestion(curIndex - 1));

    document.getElementById('testForm').addEventListener('submit', (e) => {
        e.preventDefault();

        // Final check
        if (Object.keys(userAnswers).length < questions.length) {
            const missing = questions.length - Object.keys(userAnswers).length;
            if (!confirm(`You have unanswered questions (${missing}). Submit anyway?`)) return;
        }

        submitTest();
    });

    function submitTest() {
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span> Submitting...';

        const answersPayload = Object.keys(userAnswers).map(qid => ({
            question_id: qid,
            answer: userAnswers[qid]
        }));

        fetch('test_endpoint.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                action: 'submit_test',
                material_id: materialId, // This might be wrong if materialId was session based? No, in test.php we set hidden field materialId.
                answers: answersPayload
            })
        })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    showResult(data.score);
                } else {
                    alert("Submission failed: " + data.error);
                    submitBtn.disabled = false;
                    submitBtn.textContent = 'Submit Exam';
                }
            })
            .catch(err => {
                console.error(err);
                alert("Network error.");
            });
    }

    function showResult(score) {
        testContainer.classList.add('d-none');
        document.querySelector('.d-flex.justify-content-between').classList.add('d-none'); // Hide header

        const scoreEl = document.getElementById('resultScore');
        const msgEl = document.getElementById('resultMsg');
        const titleEl = document.getElementById('resultTitle');
        const iconEl = document.getElementById('resultIcon');

        scoreEl.textContent = Math.round(score) + "%";

        if (score >= 70) {
            scoreEl.classList.add('text-success');
            titleEl.textContent = "Certification Granted";
            msgEl.textContent = "Excellent work. You have demonstrated sufficiency in this module.";
            iconEl.className = "bi bi-award-fill fs-1 text-success";
        } else {
            scoreEl.classList.add('text-danger');
            titleEl.textContent = "Review Required";
            msgEl.textContent = "You did not meet the passing criteria (70%). Please review the material and retry.";
            iconEl.className = "bi bi-x-circle fs-1 text-danger";
        }

        resultContainer.classList.remove('d-none');
    }
});
