<?php
// includes/openrouter_handler.php
// NOTE: This file now handles Google AI Studio (Gemini) API calls despite the filename.
require_once __DIR__ . '/../config.php';

/**
 * Call Google AI Studio (Gemini) API
 * @param array $messages Array of message objects [['role' => 'user', 'content' => '...']]
 * @param string $systemPrompt The personality/instruction for the bot (Optional, usually handled internally)
 * @return array|false JSON decoded response in OpenAI-compatible format (for compatibility) or false on failure
 */
function callOpenRouter($messages, $systemPrompt = null) {
    // Get OpenRouter Config
    $apiKey = defined('OPENROUTER_API_KEY') ? OPENROUTER_API_KEY : '';
    $model = defined('OPENROUTER_MODEL') ? OPENROUTER_MODEL : 'xiaomi/mimo-v2-flash:free';
    $url = defined('OPENROUTER_API_URL') ? OPENROUTER_API_URL : 'https://openrouter.ai/api/v1/chat/completions';
    
    // Build System Prompt
    if ($systemPrompt !== null) {
        $sysMsg = $systemPrompt;
    } else {
        $safeTextFull = mb_convert_encoding(defined('GLOBAL_TRAINING_INSTRUCTIONS') ? GLOBAL_TRAINING_INSTRUCTIONS : "", 'UTF-8', 'UTF-8');
        $sysMsg = "TRAINING MANUAL:\n" . mb_substr($safeTextFull, 0, 50000, 'UTF-8') . "\n\n";
        $sysMsg .= "CRITICAL INSTRUCTION: You are the Trainer Bot for DSGC (Death Squad Gaming Community).\n\n";
        $sysMsg .= "ACCURACY REQUIREMENTS:\n";
        $sysMsg .= "- Use ONLY information from the manual above\n";
        $sysMsg .= "- Quote examples EXACTLY as written in the manual\n";
        $sysMsg .= "- Do NOT add your own explanations or external knowledge\n";
        $sysMsg .= "- Do NOT change command syntax or formatting\n";
        $sysMsg .= "- Preserve all details: times (25 seconds, 300/600/900 for gag times), server IPs, reasons, etc.\n\n";
        $sysMsg .= "EXACT TEACHING SEQUENCE - ONE TOPIC PER MESSAGE:\n\n";
        $sysMsg .= "FIRST MESSAGE ONLY:\n";
        $sysMsg .= "1. Introduce DSGC (Death Squad Gaming Community) - what it is, Counter-Strike 1.6 focus\n";
        $sysMsg .= "2. List the 3 servers with their IPs (Public, AFK, Deathmatch)\n";
        $sysMsg .= "3. Explain admin responsibilities and all important rules/policies\n";
        $sysMsg .= "4. Ask if ready to learn commands, then STOP and wait for response\n\n";
        $sysMsg .= "SECOND MESSAGE (after user confirms):\n";
        $sysMsg .= "- Teach ONLY 'Setting up your Admin Password' (setinfo _pw)\n";
        $sysMsg .= "- Wait for confirmation\n\n";
        $sysMsg .= "THIRD MESSAGE:\n";
        $sysMsg .= "- Teach ONLY the Chat commands (Y@, U@, amx_say, amx_psay)\n";
        $sysMsg .= "- Wait for confirmation\n\n";
        $sysMsg .= "SUBSEQUENT MESSAGES (Basic Commands - ONE command per message):\n";
        $sysMsg .= "- amx_who (message 4)\n";
        $sysMsg .= "- amx_last (message 5)\n";
        $sysMsg .= "- amx_showip (message 6)\n";
        $sysMsg .= "- amx_kick (message 7)\n";
        $sysMsg .= "- amx_kickmenu (message 8)\n";
        $sysMsg .= "- amx_gag (message 9)\n";
        $sysMsg .= "- amx_gagmenu (message 10)\n";
        $sysMsg .= "- amx_slap (message 11)\n";
        $sysMsg .= "- amx_slapmenu (message 12)\n";
        $sysMsg .= "- amx_slay (message 13)\n";
        $sysMsg .= "- amx_nick (message 14)\n";
        $sysMsg .= "- amx_map (message 15)\n";
        $sysMsg .= "- amx_ban (message 16)\n";
        $sysMsg .= "- amx_addban (message 17)\n";
        $sysMsg .= "- amx_unban (message 18)\n";
        $sysMsg .= "- /scan (message 19)\n\n";
        $sysMsg .= "FINAL MESSAGE:\n";
        $sysMsg .= "- Teach Punishments section\n";
        $sysMsg .= "- Offer mock test ONLY after this\n\n";
        $sysMsg .= "CRITICAL RULES:\n";
        $sysMsg .= "- Do NOT combine topics in one message\n";
        $sysMsg .= "- Do NOT teach password setup or commands in the first message\n";
        $sysMsg .= "- First message = ONLY introduction + responsibilities\n";
        $sysMsg .= "- Each command = separate message with brief explanation, syntax, examples from manual\n";
        $sysMsg .= "- Wait for user confirmation before each next topic\n";
        $sysMsg .= "- Use plain text only (no bold, italic, markdown)\n";
        $sysMsg .= "- Do NOT ask 'Would you like to practice'\n";
    }

    // Prepend system message to messages array
    array_unshift($messages, ['role' => 'system', 'content' => $sysMsg]);

    // Build OpenRouter payload (OpenAI-compatible)
    $data = [
        'model' => $model,
        'messages' => $messages,
        'temperature' => 0.7,
        'max_tokens' => 4096
    ];

    // Execute Request
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: Bearer ' . $apiKey,
        'HTTP-Referer: https://dsgc-trainer.local',
        'X-Title: DSGC Admin Trainer'
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    // Handle Response
    if ($error) {
        error_log("OpenRouter Curl Error: " . $error);
        file_put_contents(__DIR__ . '/../debug_chat.log', date('Y-m-d H:i:s') . " - CURL ERROR: $error\n", FILE_APPEND);
        return false;
    }

    if ($httpCode !== 200) {
        error_log("OpenRouter API Error [$httpCode]: " . $response);
        file_put_contents(__DIR__ . '/../debug_chat.log', date('Y-m-d H:i:s') . " - API Error [$httpCode]: $response\n", FILE_APPEND);
        return false;
    }

    $json = json_decode($response, true);

    // Return OpenAI-compatible response
    if (isset($json['choices'][0]['message']['content'])) {
        file_put_contents(__DIR__ . '/../debug_chat.log', date('Y-m-d H:i:s') . " - SUCCESS with OpenRouter\n", FILE_APPEND);
        return $json;
    }

    return false;
}

// Helper functions (unused now but kept for compatibility logic if needed)
function formatBotResponse($apiResponse) {
    if (!$apiResponse || !isset($apiResponse['choices'][0]['message']['content'])) {
        return "I apologize, but I'm having trouble connecting to my brain right now. Please try again.";
    }
    return $apiResponse['choices'][0]['message']['content'];
}

// buildTrainingContext is largely redundant now but we keep it to return the 'messages' array
function buildTrainingContext($session, $extractedText, $recentChatHistory) {
    // Only return user/bot history. System prompt is handled in callOpenRouter.
    $messages = [];
    foreach ($recentChatHistory as $chat) {
        $messages[] = [
            'role' => ($chat['sender'] == 'bot') ? 'assistant' : 'user',
            'content' => $chat['message']
        ];
    }
    return ['messages' => $messages, 'systemPrompt' => ''];
}

function getSystemPrompt($skillLevel) { return ""; }
?>
